/**
* @class CCPCDisc
* Classe permettant la gestion d'un disc CPC (utilisation de la sous-lib DSK)
* @author Thierry JOUIN
* @version 1.1
* @date 31/10/2001
*/

#pragma warning( disable : 4786 )
#include <string>
#include <iostream>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "libdsk.h"

#include <map>
#include <vector>

#include <math.h>

#include "CCPCFile.h"

#ifndef _CCPCDISC_HH_
#define _CCPCDISC_HH_

enum TDisc
{
    Data,
		System
};

struct SDiscFileEntryKey
{
	unsigned char user;
	char name[11];
	bool writeProtected;
	bool system;
	bool operator()(const SDiscFileEntryKey &i_key1,const SDiscFileEntryKey &i_key2) const;
	static void SDiscFileEntryKey::convertStringToSDiscFileEntryKey(const std::string &i_name, SDiscFileEntryKey &o_file);
};

struct SDiscFileEntry
{
	unsigned int size;
	unsigned int nbRecord;
	std::vector<unsigned int> blocks;
};

class CCPCDisc
{
private:
protected:
	/// Driver libDSK utilise
	DSK_PDRIVER _driver;
	/// Geometrie du disc
	DSK_GEOMETRY _geometry;
	
	/// Table contenant toutes les entree du catalogue
	std::map<SDiscFileEntryKey,SDiscFileEntry,SDiscFileEntryKey> _catalogue;
	/// Catalogue buffer, contient les donnees version calatalogue du disc courant
	char *_catalogueBuffer;
	/// Indique si le catalogue a change
	bool _catalogueChanged;
	
	/// Affiche l'occupation du disc
	virtual void scanBlock(std::ostream &io_os) const = 0;
	/// Renvoie la liste des block libres
	virtual void getEmptyBlock(std::vector<unsigned int> &o_block) const = 0;
	
	/// Pour une entree du catalogue, affiche la position des donnees sur le disc
	virtual void printBlockForEntry(std::ostream &io_os, unsigned int i_entry) const = 0;
	
	/// Lit le buffer catalogue sur le disc
	virtual void readCatalogueBuffer() = 0;
	/// Ecrit le buffer catalogue sur le disc
	virtual void writeCatalogueBuffer() = 0;
	
	/// Lit toutes les entrees du catalogue, et les mets dans la structure catalogue
	void readCatalogue();
	/// Ecrit toutes les dans le buffer et ecrit sur le disc
	void writeCatalogue();
	
	/// Ajoute une entre dans le catalogue (ne sauvegarde pas le catalogue !)
	void addCatalogueEntry(const SDiscFileEntryKey &i_name, const SDiscFileEntry &i_nameData);
	/// Enleve une entre du catalogue
	void removeCatalogueEntry(const SDiscFileEntryKey &i_name);
	
	/// Renvoie le nombre d'entre utilise dans le catalogue courant
	int getNbUsedEntry() const;
	
	/// Renvoie l'entree du catalogue qui contient le block passe en parametre (-1 quand non trouve)
	int getCatalogueEntryForBlock(unsigned int i_block) const;
	
public:
	
	/// Constructeur
	CCPCDisc();
	/// Constructeur, on connait le driver et la gomtrie
	CCPCDisc(DSK_PDRIVER i_driver, const DSK_GEOMETRY &i_geometry);
	/// Destructeur
	virtual ~CCPCDisc();
	
	/// Ouverture d'un DSK existant
	static CCPCDisc* openDisc(const std::string &i_filename, int i_inside = 0);
	/// Creation (et formattage) d'un DSK existant
	static CCPCDisc* createDisc(const std::string &i_filename, const TDisc &i_type, int i_inside = 0);
	
	/// Ouverture d'un DSK deja existant
	virtual void open(const std::string &i_filename, int i_inside = 0) = 0;
	/// Ouverture en creation d'un DSK
	virtual void create(const std::string &i_filename, int i_inside = 0) = 0;
	/// Fermeture d'un DSK
	virtual void close();
	
	/// Formattage du DSK
	virtual void format() =0;
	
	/// Renvoie le nombre de fichier sur le disc
	unsigned int getNbFiles() const;
	/// Renvoie le nom du fichier dont on passe l'indice
	std::string getFilename(const unsigned int i_id) const;
	
	/// Affiche le catalogue du disc dans une stream
	void catalogue(std::ostream &io_os) const;
	
	/// Affiche les infos sur le disc
	void printInfo(std::ostream &io_os) const;
	
	/// Renvoie le fichier dont on passe le nom en parametre
	CCPCFile* getFile(const std::string &i_filename) const;
	/// Ajoute un fichier sur le disc
	void putFile(const CCPCFile &i_file, const std::string &i_filename);
	
	/// Efface un fichier sur le disc
	void eraseFile(const std::string &i_filename);
	
	/// Renome un fichier sur le disc
	void renameFile(const std::string &i_oldFilename, const std::string &i_newFilename);
	
	/// Lit un secteur sur le disc (buffer deja alloue !)
	virtual void readSector(const unsigned char i_cylinder, const unsigned char i_head, const unsigned char i_sectorID, void* i_buffer) const = 0;
	/// Ecrit un secteur sur le disc
	virtual void writeSector(const unsigned char i_cylinder, const unsigned char i_head, const unsigned char i_sectorID, const void* i_buffer) = 0;
	/// Lit un bloc sur le disc (buffer deja alloue !)
	virtual void readBlock(const unsigned char i_idBlock, void* i_buffer) const = 0;
	/// Ecrit un secteur sur le disc
	virtual void writeBlock(const unsigned char i_idBlock, const void* i_buffer) = 0;
	
	/// Indique qu'il n'y a pas de block
	static unsigned int no_block;
	
};

#endif
