/**
* @class CCPCDisc
* Classe permettant la gestion d'un disc CPC (utilisation de la sous-lib DSK)
* @author Thierry JOUIN
* @version 1.1
* @date 31/10/2001
*/

#include "CCPCDisc.h"

#include "CCPCDataDisc.h"
#include "CCPCSystemDisc.h"

#include "CError.h"

#ifdef WIN32
#define FILE_SEPARATOR '\\'
#else
#define FILE_SEPARATOR '/'
#endif

char* CCPCDisc::TDiscDesc[] = 
{
	"cpcdata",
	"cpcdata42",
	"cpcsys",
	"cpcsys42",
	"parados80"
};

unsigned int CCPCDisc::no_block = (0-1);

// CPC file entry key methods

CCPCDisc::CDiscFileEntryKey::CDiscFileEntryKey() :
User(0),
WriteProtected(false),
System(false)
{
	memset(Name,11,1);
}

CCPCDisc::CDiscFileEntryKey::CDiscFileEntryKey(const std::string &i_name) :
User(0),
WriteProtected(false),
System(false)
{
	convertStringToFilename(i_name);
}

std::string CCPCDisc::CDiscFileEntryKey::getFilename() const
{
	int i=0;
	std::string filename;
	while (i<8 && Name[i] != ' ')
    {
		filename += Name[i++]; 
    }
	i = 8;
	filename += '.';
	while (i<11 && Name[i] != ' ')
		filename += Name[i++];
	
	return filename;
}

void CCPCDisc::CDiscFileEntryKey::convertStringToFilename(const std::string &i_name)
{
	unsigned int i;
	unsigned int p = i_name.find_last_of('.');
	
	std::string n = i_name;
	std::string e;
	if (n.find(FILE_SEPARATOR) != std::string::npos)
    {
		n = n.substr(n.find_last_of(FILE_SEPARATOR)+1,n.size() - n.find_last_of(FILE_SEPARATOR));
    }
	if (n.find('.') != std::string::npos)
	{
		e = n.substr(n.find('.')+1, n.size() - n.find('.'));
		n = n.substr(0, n.find('.'));
	}
	for (i=0;i<8;i++)
	{
		if (i<n.size())
			Name[i]=toupper(n[i]);
		else
			Name[i]=' ';
	}
	for (i=0;i<3;i++)
	{
		if (i<e.size())
			Name[i+8]=toupper(e[i]);
		else
			Name[i+8]=' ';
	}
}

bool CCPCDisc::CDiscFileEntryKey::operator()(const CDiscFileEntryKey &i_file1,const CDiscFileEntryKey &i_file2) const
{
	return ( (i_file1.User < i_file2.User) || ((i_file1.User == i_file2.User) && (strncmp(i_file1.Name,i_file2.Name,11) < 0)) );
}

// CPC file entry methods

CCPCDisc::CDiscFileEntry::CDiscFileEntry() :
Size(0),
NbRecord(0),
Blocks()
{
}

// CPC Disc methods

CCPCDisc::CCPCDisc()
:_driver(NULL),
_catalogue(),
_catalogueBuffer(NULL),
_catalogueChanged(false)
{
}

CCPCDisc::CCPCDisc(DSK_PDRIVER i_driver, const TDisc format)
:_driver(i_driver),
_format(format),
_catalogue(),
_catalogueBuffer(),
_catalogueChanged(false)
{
	dsk_err_t e;
	
	TOOLS_ASSERTMSG( (format >= 0 && format < maxTDisc) , "Error opening dsk : unknown format");

	dsk_format_t dsk_format = getLibDskFormat(TDiscDesc[format]);
	
	e = dg_stdformat(&_geometry, dsk_format, NULL, NULL);
	TOOLS_ASSERTMSG( (e==DSK_ERR_OK) , "Error opening dsk :" << std::string(dsk_strerror(e)));
	
	_catalogueBuffer = new char[_geometry.dg_secsize*4];
}

CCPCDisc::~CCPCDisc()
{
	if (_catalogueBuffer != NULL)
		delete[] _catalogueBuffer;
}

bool CCPCDisc::isDSK(const std::string &i_filename, int i_inside)
{
	DSK_PDRIVER driver;
	dsk_err_t e;
	
	bool isDSK = false;
	e = dsk_open(&driver,i_filename.c_str(), NULL, NULL);
	if (e == DSK_ERR_OK)
	{
		/*
		e = dsk_set_forcehead(driver, i_inside);
		TOOLS_ASSERTMSG( (e==DSK_ERR_OK) , "Error opening dsk :" << std::string(dsk_strerror(e)));
		*/
		
		TDisc format;
		dsk_format_t dsk_format;
		
		format = guessGeometry(driver, dsk_format);

		isDSK = (format != -1);

		dsk_close(&driver);
	}
	
	return isDSK;
}

dsk_format_t CCPCDisc::getLibDskFormat(const std::string &name)
{
	dsk_cchar_t fname;
	dsk_format_t fmt = FMT_180K;
	while (dg_stdformat(NULL, fmt, &fname, NULL) == DSK_ERR_OK)
	{
			if (!strcmpi(name.c_str(), fname)) 
				return fmt;
			fmt = (dsk_format_t)(fmt+1);
	}
	return (dsk_format_t)-1;
}

CCPCDisc::TDisc CCPCDisc::guessGeometry(DSK_PDRIVER self, dsk_format_t &dsk_format, bool extendedGuess)
{
	DSK_GEOMETRY candidateGeom;
	TDisc candidateFormat = (TDisc)-1;
	
	for (int f=0 ; f < maxTDisc ; f++)
	{
		dsk_format = getLibDskFormat(TDiscDesc[f]);
		if (dsk_format != -1)
		{
			DSK_GEOMETRY geom;
			DSK_FORMAT sectorId;
			dsk_err_t e;

			dg_stdformat(&geom, dsk_format, NULL, NULL);

			// First check sector id reading
			e = dsk_psecid(self, &geom, 0, 0, &sectorId);
			if (e == DSK_ERR_OK && 
				sectorId.fmt_sector >= geom.dg_secbase &&
				sectorId.fmt_sector < (geom.dg_secbase + geom.dg_sectors))
			{
				// We found a right sector, if no candidate already found,
				// we use this format as a start (even if number of track is not good !)
				if (candidateFormat == -1)
				{
					candidateGeom = geom;
					candidateFormat = (TDisc)f;
				}

				if (extendedGuess)
				{
					// if found, check last track sector id reading
					e = dsk_psecid(self, &geom, geom.dg_cylinders-1, 0, &sectorId);
					if (e == DSK_ERR_OK && 
						sectorId.fmt_sector >= geom.dg_secbase &&
						sectorId.fmt_sector < (geom.dg_secbase + geom.dg_sectors))
					{
						if ((candidateFormat == -1) ||
							((candidateFormat != -1) && (candidateGeom.dg_cylinders < geom.dg_cylinders)))
						{
							candidateGeom = geom;
							candidateFormat = (TDisc)f;
						}
					}
				}
			}
		}
	}

	return candidateFormat;
}

CCPCDisc* CCPCDisc::openDisc(const std::string &i_filename, int i_inside)
{
	CCPCDisc* disc = NULL;
	DSK_PDRIVER driver;
	dsk_err_t e;
	
	e = dsk_open(&driver,i_filename.c_str(), NULL, NULL);
	TOOLS_ASSERTMSG( (e==DSK_ERR_OK) , "Error opening dsk :" << std::string(dsk_strerror(e)));
	
	/*
	e = dsk_set_forcehead(driver, i_inside);
	TOOLS_ASSERTMSG( (e==DSK_ERR_OK) , "Error opening dsk :" << std::string(dsk_strerror(e)));
	*/
	
	TDisc format;
	dsk_format_t dsk_format;
	
	format = guessGeometry(driver, dsk_format);

	TOOLS_ASSERTMSG( (format != -1) , "Error opening dsk : unknown geometry");	

	switch (format)
    {
    case Data:
	case Data42:
		{
			disc = new CCPCDataDisc(driver, format);
			disc->readCatalogue();
			break;
		}
    case System:
	case System42:
		{
			disc = new CCPCSystemDisc(driver, format);
			disc->readCatalogue();
			break;
		}
    default:
		{
			TOOLS_ERRORMSG( "Error opening dsk : unknown geometry");
		}
    }
	
	return disc;
}

CCPCDisc* CCPCDisc::createDisc(const std::string &i_filename, const TDisc &i_type, int i_inside)
{
	CCPCDisc *d=NULL;
	switch (i_type)
    {
    case Data:
	case Data42:
		{
			d = new CCPCDataDisc();
			d->create(i_type, i_filename, i_inside);
			break;
		}
    case System:
	case System42:
		{
			d = new CCPCDataDisc();
			d->create(i_type, i_filename, i_inside);
			break;
		}
    }
	return d;
}

void CCPCDisc::close()
{
	if (_catalogueChanged)
		writeCatalogue();
	dsk_close(&_driver);
}

void CCPCDisc::readCatalogue()
{
	// On charge le buffer catalogue
	readCatalogueBuffer();
	
	char *currentEntry = _catalogueBuffer;
	// On parcours toutes les entrees du catalogue
	for (unsigned int e=0;e<64;e++)
    {
		CDiscFileEntryKey name;
		CDiscFileEntry nameValues;
		
		name.User = (unsigned char) currentEntry[0];
		
		strncpy(name.Name,(char*)currentEntry+1,11);
		
		// On recupere les bit d'indications
		name.WriteProtected = ((name.Name[8] & 128) != 0);
		name.System = ((name.Name[9] & 128) != 0);
		
		bool validName = true;
		
		// On 'nettoie' le nom
		for (unsigned int i=0;i<11;i++)
			validName = validName && ((name.Name[i] & 127)>=' ') && (name.Name[i]!=(char)229);
		
		if (validName)
		{
			std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::iterator pos = _catalogue.find(name);
			
			for (unsigned int i=0;i<11;i++)
				name.Name[i]=name.Name[i] & 127;
			
			if (pos == _catalogue.end())
			{
				nameValues.Blocks = std::vector<unsigned int>(256,CCPCDisc::no_block);
				nameValues.Size = 0;
				nameValues.NbRecord = 0;
				_catalogue[name]=nameValues;
			}
			
			unsigned char ordreChargement = currentEntry[12];
			
			for(unsigned int b=16;b<32;b++)
			{
				if (currentEntry[b] != 0)
					if (_catalogue[name].Blocks[ordreChargement*16+(b-16)] == CCPCDisc::no_block || name.User == 229)
					{
						_catalogue[name].Blocks[ordreChargement*16+(b-16)]=currentEntry[b];
						_catalogue[name].Size++;
					}
					else
					{
						PRINTVAR(std::string(name.Name,11));
						TOOLS_ERRORMSG("Invalid entry order !");
					}
			}
			_catalogue[name].NbRecord += currentEntry[0xf];
		}
		currentEntry += 32;
    }
}

void CCPCDisc::writeCatalogue()
{
	if (_catalogueChanged)
    {
		char* pCatBuffer = _catalogueBuffer;
		
		// On vide le catalogue courant
		for (unsigned int i=0;i<_geometry.dg_secsize*4;i++)
			_catalogueBuffer[i]=(char)0xe5;
		
		// On en recre un nouveau
		for (std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator it=_catalogue.begin();it!=_catalogue.end();it++)
		{
			int nbEntry = (int)ceil(((float)it->second.Size)/16.0);
			char name[11];
			char ordreChargement=0;
			unsigned int nbRecord = it->second.NbRecord;
			unsigned int nbRecordMaxPerEntry = (_geometry.dg_secsize*2*16)/128;
			
			memcpy(name,it->first.Name,11);
			name[8] = it->first.WriteProtected ? name[8]|128 : name[8];
			name[9] = it->first.System ? name[9]|128 : name[9];
			for (int j=0;j<nbEntry;j++)
			{
				for (unsigned int k=0;k<32;k++)
					pCatBuffer[k]=0;
				
				pCatBuffer[0] = (char)it->first.User;
				memcpy(pCatBuffer+1,name,11);
				pCatBuffer[12] = ordreChargement;
				pCatBuffer[15] = (nbRecord > nbRecordMaxPerEntry) ? nbRecordMaxPerEntry : nbRecord;
				for (unsigned int b=16;b<32;b++)
				{
					unsigned int bI = ordreChargement*16+b-16;
					if (bI < it->second.Size)
						pCatBuffer[b] = it->second.Blocks[bI];
				}
				pCatBuffer += 32;
				ordreChargement++;
				nbRecord -= nbRecordMaxPerEntry;
			}
			//io_os << (unsigned int)it->first.User << ":" << std::string(it->first.Name,0,11) << " " << it->second.Size << "Kb" << std::endl;
		}
		writeCatalogueBuffer();
    }
}

void CCPCDisc::addCatalogueEntry(const CDiscFileEntryKey &i_name, const CDiscFileEntry &i_nameData)
{
	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::iterator iNorm = _catalogue.find(i_name);
	
	int nbEntry = (int)ceil(((float)i_nameData.Size)/16.0);
	
	TOOLS_ASSERTMSG( (nbEntry < (64-getNbUsedEntry())) , "Directory Full");
	
	if (iNorm != _catalogue.end())
    {
		CDiscFileEntryKey name_bak = i_name;
		name_bak.Name[8]='B';name_bak.Name[9]='A';name_bak.Name[10]='K';
		
		std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::iterator iBak = _catalogue.find(name_bak);
		
		if (iBak != _catalogue.end())
		{
			CDiscFileEntryKey name_bak_del = name_bak;
			name_bak_del.User=229;
			
			_catalogue[name_bak_del] = _catalogue[name_bak];
		}
		_catalogue[name_bak]=_catalogue[i_name];
    }
	_catalogue[i_name]=i_nameData;  
	
	_catalogueChanged = true;
	
}
void CCPCDisc::removeCatalogueEntry(const CDiscFileEntryKey &i_name)
{
	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::iterator iNorm = _catalogue.find(i_name);
	
	if (iNorm != _catalogue.end())
    {
		CDiscFileEntryKey name_del = i_name;
		name_del.User = 229;
		_catalogue[name_del]=_catalogue[i_name];
		_catalogue.erase(iNorm);
    }
	else
    {
		WARNING("File not in the catalogue !");
    }
	_catalogueChanged = true;
}

int CCPCDisc::getNbUsedEntry() const
{
	int nbUsed = 0;
	for (std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator i=_catalogue.begin();i!=_catalogue.end();i++)
    {
		nbUsed+=(int)ceil(((float)i->second.Size)/16.0);
    }
	return nbUsed;
}

int CCPCDisc::getCatalogueEntryForBlock(unsigned int i_block) const
{
	int entry = -1;
	int numEntry = 0;
	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator e = _catalogue.begin();
	while (e!=_catalogue.end() && entry == -1)
    {
		if (e->first.User != 229)
			for (unsigned int i=0;i<e->second.Blocks.size();i++)
				if (e->second.Blocks[i] == i_block)
					entry = numEntry;
				e++;
				numEntry++;
    }
	return entry;
}

unsigned int CCPCDisc::getNbFiles() const
{
	return _catalogue.size();
}
std::string CCPCDisc::getFilename(const unsigned int i_id, int &user) const
{
	unsigned int i;
	TOOLS_ASSERTMSG( (i_id<_catalogue.size()), "");
	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator e = _catalogue.begin();
	for (i=0 ; i<i_id ; i++)
		e++;

	user = e->first.User;
	return (e->first.getFilename());
}
const CCPCDisc::CDiscFileEntryKey& CCPCDisc::getFileEntry(const unsigned int i_id) const
{
	unsigned int i;
	TOOLS_ASSERTMSG( (i_id<_catalogue.size()), "");
	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator e = _catalogue.begin();
	for (i=0 ; i<i_id ; i++)
		e++;
	return e->first;
}
int CCPCDisc::getFileSize(const unsigned int i_id) const
{
	unsigned int i;
	TOOLS_ASSERTMSG( (i_id<_catalogue.size()), "");
	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator e = _catalogue.begin();
	for (i=0 ; i<i_id ; i++)
		e++;
	return e->second.Size;
}
void CCPCDisc::catalogue(std::ostream &io_os) const
{
	io_os << "Catalogue : " << _catalogue.size() << std::endl;
	
	for (std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator i=_catalogue.begin();i!=_catalogue.end();i++)
    {
		io_os.width(3);
		io_os << (unsigned int)i->first.User << ":" << std::string(i->first.Name,0,8) << "." << std::string(i->first.Name,8,3) << " ";
		if (i->first.System)
		{
			io_os << "S";
		}
		else
		{
			io_os << " ";
		}
		if (i->first.WriteProtected)
		{
			io_os << "*";
		}
		else
		{
			io_os << " ";
		}
		io_os << i->second.Size << "Kb" << std::endl;
		io_os.width(0);
    }
}

void CCPCDisc::printInfo(std::ostream &io_os) const
{
	io_os << "Disc info : " << std::endl;
	
	scanBlock(io_os);
	
	std::vector<unsigned int> emptyBlocks;
	getEmptyBlock(emptyBlocks);
	io_os << "Free : " << emptyBlocks.size() << std::endl;
	for (unsigned int i=0;i<emptyBlocks.size();i++)
    {
		io_os << emptyBlocks[i] << ",";
    }
	io_os << std::endl;
}

int CCPCDisc::getDiscCapacity() const
{
	int capacity = _geometry.dg_cylinders * _geometry.dg_secsize * _geometry.dg_sectors;

	capacity -= _geometry.dg_secsize*4;

	capacity = capacity >> 10;

	return capacity;
}
const DSK_GEOMETRY& CCPCDisc::getDiscGeometry() const
{
	return _geometry;
}

std::string CCPCDisc::getDiscFormatName() const
{
	std::string name = "";
	dsk_format_t dsk_format = getLibDskFormat(TDiscDesc[_format]);
	if (dsk_format != -1)
	{
		dsk_cchar_t dskName;
		dg_stdformat(NULL, dsk_format, &dskName, NULL);
		name = dskName;
	}
	return name;
}

std::string CCPCDisc::getDiscFormatDescription() const
{
	std::string desc = "";
	dsk_format_t dsk_format = getLibDskFormat(TDiscDesc[_format]);
	if (dsk_format != -1)
	{
		dsk_cchar_t dskDesc;
		dg_stdformat(NULL, dsk_format, NULL, &dskDesc);
		desc = dskDesc;
	}
	return desc;
}

CCPCFile* CCPCDisc::getFile(const std::string &i_filename, int user) const
{
	CDiscFileEntryKey key(i_filename);
	
	key.User = user;

	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::const_iterator pos = _catalogue.find(key);
	
	TOOLS_ASSERTMSG( (pos!= _catalogue.end()) , "Unknown filename " << i_filename);  
	
	unsigned char *file = new unsigned char[pos->second.Size*1024];
	for (unsigned int i=0;i<pos->second.Size;i++)
		readBlock(pos->second.Blocks[i],file+(1024*i));

	CCPCFile *fileOpen = CCPCFile::openFile(file,pos->second.Size*1024);
	delete[] file;
	return fileOpen;
}

void CCPCDisc::putFile(const CCPCFile &i_file, const std::string &i_filename, int user, bool system, bool writeProtected)
{
	std::vector<unsigned int> emptyBlock;
	getEmptyBlock(emptyBlock);
	
	unsigned int neededBlock = (unsigned int)ceil((double)i_file.getDatasSize() / (double)(_geometry.dg_secsize*2));
	unsigned int neededRecord = (unsigned int)ceil((double)i_file.getDatasSize() / (double)(128));
	
	int emptyBlockSize = emptyBlock.size();
	TOOLS_ASSERTMSG( (neededBlock <= emptyBlock.size()) , "Unable to add file to disc - Disc Full");
	
	unsigned char *fileBuffer = i_file.getDatas();
	unsigned int fileBufferSize = i_file.getDatasSize();
	
	unsigned int blockSize = _geometry.dg_secsize*2;

	unsigned char *block = new unsigned char[blockSize];
	
	CDiscFileEntryKey fileEntryKey(i_filename);
	CDiscFileEntry fileEntry;

	fileEntryKey.User = user;
	fileEntryKey.System = system;
	fileEntryKey.WriteProtected = writeProtected;
	
	fileEntry.Size = neededBlock;
	fileEntry.NbRecord = neededRecord;
	fileEntry.Blocks.clear();
	
	int remainingBytes = fileBufferSize;
	for (unsigned int j=0;j<neededBlock;j++)
    {
		int copySize = (remainingBytes > blockSize) ? blockSize : remainingBytes;

		memset(block,0,blockSize);
		
		memcpy(block,fileBuffer+(j*blockSize),copySize);
		
		writeBlock(emptyBlock[j],block);
		
		fileEntry.Blocks.push_back(emptyBlock[j]);

		remainingBytes -= blockSize;
    }
	
	
	addCatalogueEntry(fileEntryKey,fileEntry);
	delete[] block;
}

/// Efface un fichier sur le disc
void CCPCDisc::eraseFile(const std::string &i_filename, int user)
{
	CDiscFileEntryKey key(i_filename);
	
	key.User = user;
	
	removeCatalogueEntry(key);
	
}

/// Renome un fichier sur le disc
void CCPCDisc::renameFile(	const std::string &i_oldFilename, const std::string &i_newFilename,
							int old_user, int new_user,
							bool old_sys, bool new_sys,
							bool old_pro, bool new_pro)
{
	CDiscFileEntryKey from(i_oldFilename);
	CDiscFileEntryKey to(i_newFilename);
	
	from.User = old_user;
	from.System = old_sys;
	from.WriteProtected = old_pro;

	to.User = new_user;
	to.System = new_sys;
	to.WriteProtected = new_pro;

	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey>::iterator iFrom = _catalogue.find(from);
	TOOLS_ASSERTMSG( iFrom != _catalogue.end() , "Unknown file " << i_oldFilename);
	
	addCatalogueEntry(to,_catalogue[from]);
	
	_catalogue.erase(iFrom);
	
}
