/**
* @class CCPCDisc
* Class used for manipulating generic libdsk structure
* @author Thierry JOUIN
*/

#pragma warning( disable : 4786 )
#include <string>
#include <iostream>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "libdsk.h"

#include <map>
#include <vector>

#include <math.h>

#include "CCPCFile.h"

#ifndef _CCPCDISC_HH_
#define _CCPCDISC_HH_

//! General CPC disc class manager
class CCPCDisc
{
public:
	//! Enum type of 
	enum TDisc
	{
		Data = 0,
		Data42,
		System,
		System42,
		Parados80,
		maxTDisc
	};

	//! Define a key in directory structure
	class CDiscFileEntryKey
	{
	public:
		//! File user
		unsigned char User;
		//! File name
		char Name[11];
		//! File write protected
		bool WriteProtected;
		//! System file
		bool System;
	public:
		//! Default constructor
		CDiscFileEntryKey();
		//! Constructor using a standard name (adjust to Name[11])
		CDiscFileEntryKey(const std::string &i_name);

		//! Comparaison operator (used for map)
		bool operator()(const CDiscFileEntryKey &i_key1,const CDiscFileEntryKey &i_key2) const;

		//! Return formatted filename for file entry
		std::string getFilename() const;
	protected:
		//! Convert a string to standard CPC filename
		void convertStringToFilename(const std::string &i_name);
	};

	//! Define a entry in directory
	class CDiscFileEntry
	{
	public:
		//! Size of the file
		unsigned int Size;
		//! Number of record used by file
		unsigned int NbRecord;
		//! List of block index on disc
		std::vector<unsigned int> Blocks;

	public:
		//! Default constructor
		CDiscFileEntry();
	};

protected:
	//! Name of all cpc format (use to find geometry in libdsk)
	static char* TDiscDesc[];
	//! There is no block
	static unsigned int no_block;

protected:
	//! Driver libDSK utilise
	DSK_PDRIVER _driver;
	//! Geometrie du disc
	DSK_GEOMETRY _geometry;
	//! CPC DSK format
	TDisc _format;

	//! Table contenant toutes les entree du catalogue
	std::map<CDiscFileEntryKey,CDiscFileEntry,CDiscFileEntryKey> _catalogue;
	//! Catalogue buffer, contient les donnees version calatalogue du disc courant
	char *_catalogueBuffer;
	//! Indique si le catalogue a change
	bool _catalogueChanged;
	
protected:
	//! Try to detect geometry of current driver & return format
	static TDisc guessGeometry(DSK_PDRIVER self, dsk_format_t &dsk_format, bool extendedGuess = true);
	//! Get back libdsk index of a given format name
	static dsk_format_t getLibDskFormat(const std::string &name);

	//! Affiche l'occupation du disc
	virtual void scanBlock(std::ostream &io_os) const = 0;
	//! Renvoie la liste des block libres
	virtual void getEmptyBlock(std::vector<unsigned int> &o_block) const = 0;
	
	//! Pour une entree du catalogue, affiche la position des donnees sur le disc
	virtual void printBlockForEntry(std::ostream &io_os, unsigned int i_entry) const = 0;
	
	//! Lit le buffer catalogue sur le disc
	virtual void readCatalogueBuffer() = 0;
	//! Ecrit le buffer catalogue sur le disc
	virtual void writeCatalogueBuffer() = 0;
	
	//! Lit toutes les entrees du catalogue, et les mets dans la structure catalogue
	void readCatalogue();
	//! Ecrit toutes les dans le buffer et ecrit sur le disc
	void writeCatalogue();
	
	//! Ajoute une entre dans le catalogue (ne sauvegarde pas le catalogue !)
	void addCatalogueEntry(const CDiscFileEntryKey &i_name, const CDiscFileEntry &i_nameData);
	//! Enleve une entre du catalogue
	void removeCatalogueEntry(const CDiscFileEntryKey &i_name);
	
	//! Renvoie le nombre d'entre utilise dans le catalogue courant
	int getNbUsedEntry() const;
	
	//! Renvoie l'entree du catalogue qui contient le block passe en parametre (-1 quand non trouve)
	int getCatalogueEntryForBlock(unsigned int i_block) const;
	
public:
	
	//! Constructeur
	CCPCDisc();
	//! Constructeur, on connait le driver et la gomtrie
	CCPCDisc(DSK_PDRIVER i_driver, const TDisc format);
	//! Destructeur
	virtual ~CCPCDisc();
	
	//! Check if file is a DSK
	static bool isDSK(const std::string &i_filename, int i_inside = 0);
	//! Ouverture d'un DSK existant
	static CCPCDisc* openDisc(const std::string &i_filename, int i_inside = 0);
	//! Creation (et formattage) d'un DSK existant
	static CCPCDisc* createDisc(const std::string &i_filename, const TDisc &i_type, int i_inside = 0);
	
	//! Ouverture d'un DSK deja existant
	virtual void open(const std::string &i_filename, int i_inside = 0) = 0;
	//! Ouverture en creation d'un DSK
	virtual void create(TDisc i_type, const std::string &i_filename, int i_inside = 0) = 0;
	//! Fermeture d'un DSK
	virtual void close();
	
	//! Formattage du DSK
	virtual void format(bool interlaced = true) =0;
	
	//! Renvoie le nombre de fichier sur le disc
	unsigned int getNbFiles() const;
	//! Renvoie le nom du fichier dont on passe l'indice
	std::string getFilename(const unsigned int i_id, int &user) const;
	//! Get file entry
	const CDiscFileEntryKey& getFileEntry(const unsigned int i_id) const;
	//! Get file entry file size
	int getFileSize(const unsigned int i_id) const;
	
	//! Affiche le catalogue du disc dans une stream
	void catalogue(std::ostream &io_os) const;
	
	//! Affiche les infos sur le disc
	void printInfo(std::ostream &io_os) const;
	
	//! Return disc capacity
	int getDiscCapacity() const;
	//! Return disc geometry
	const DSK_GEOMETRY& getDiscGeometry() const;
	//! Return format name
	std::string getDiscFormatName() const;
	//! Return format description
	std::string getDiscFormatDescription() const;

	//! Renvoie le fichier dont on passe le nom en parametre
	CCPCFile* getFile(const std::string &i_filename, int user = 0) const;
	//! Ajoute un fichier sur le disc
	void putFile(const CCPCFile &i_file, const std::string &i_filename, int user = 0, bool system = false, bool writeProtected = false);
	
	//! Efface un fichier sur le disc
	void eraseFile(const std::string &i_filename, int user = 0);
	
	//! Renome un fichier sur le disc
	void renameFile(const std::string &i_oldFilename, const std::string &i_newFilename,
					int old_user = 0, int new_user = 0,
					bool old_sys = false, bool new_sys = false,
					bool old_pro = false, bool new_pro = false);
	
	//! Lit un secteur sur le disc (buffer deja alloue !)
	virtual void readSector(const unsigned char i_cylinder, const unsigned char i_head, const unsigned char i_sectorID, void* i_buffer) const = 0;
	//! Ecrit un secteur sur le disc
	virtual void writeSector(const unsigned char i_cylinder, const unsigned char i_head, const unsigned char i_sectorID, const void* i_buffer) = 0;
	//! Lit un bloc sur le disc (buffer deja alloue !)
	virtual void readBlock(const unsigned char i_idBlock, void* i_buffer) const = 0;
	//! Ecrit un secteur sur le disc
	virtual void writeBlock(const unsigned char i_idBlock, const void* i_buffer) = 0;
};

#endif
